/*
 * Color Collapse
 * Copyright (c) 1998, 2002, 2004-2006 Blazing Games Inc. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package com.blazinggames.cc;

import java.io.*;

/**
 * Holds the collapse game grid.
 * 
 * @author Billy D. Spelchan
 * @version 1.0 OSR
 */
class Grid {
// *** CONSTANTS ***
	public static final int clpsNUMCOLS = 20;
	public static final int clpsNUMROWS = 20;

// *** VARIABLES ***
	byte m_nGrid[][] = new byte[clpsNUMCOLS][clpsNUMROWS];
	byte m_nRefresh[][] = new byte[clpsNUMCOLS][clpsNUMROWS];
			// Grid data. [Col (Left to Right)][Row (Bottom to Top)]

// *** CONSTRUCTOR ***
	// The constructor
	Grid()	{
		clear();
	}

// *** GRID MANIPULATION ***
	// Clears grid
	void clear() {
		int cntrC, cntrR;

		for (cntrC = 0; cntrC < clpsNUMCOLS; ++cntrC)
			for (cntrR = 0; cntrR < clpsNUMROWS; ++cntrR) {
				m_nGrid[cntrC][cntrR] = 0;
				m_nRefresh[cntrC][cntrR] = 0;
			}
	}

	// Copies grid into refresh grid
	void storeGrid() {
		int cntrC, cntrR;

		for (cntrC = 0; cntrC < clpsNUMCOLS; ++cntrC)
			for (cntrR = 0; cntrR < clpsNUMROWS; ++cntrR)
				m_nRefresh[cntrC][cntrR] = m_nGrid[cntrC][cntrR];
	}

	// Copies refresh grid into grid
	void refreshGrid() {
		int cntrC, cntrR;

		for (cntrC = 0; cntrC < clpsNUMCOLS; ++cntrC)
			for (cntrR = 0; cntrR < clpsNUMROWS; ++cntrR)
				m_nGrid[cntrC][cntrR] = m_nRefresh[cntrC][cntrR];
	}

	// Fully collapses the blocks on the grid
	void fullCollapse() {
		int nIsFalling, cntrC;

		do {
			nIsFalling = 0;

			for (cntrC = 0; cntrC < clpsNUMCOLS; ++cntrC)
				nIsFalling += collapseColumn(cntrC);
		} while (nIsFalling > 0);
	}

	// recursively removes chain of blocks
	void removeBlock(int nCol, int nRow) {
		byte nBlock;

		// First, safty check to prevent infinite recursion
		if (m_nGrid[nCol][nRow] == 0)
			return;

		nBlock = m_nGrid[nCol][nRow];
		m_nGrid[nCol][nRow] = 0;
		// Check above
		if (nRow < (clpsNUMROWS - 1))
			if (m_nGrid[nCol][nRow + 1] == nBlock)
				removeBlock(nCol, nRow + 1);
		// Check below
		if (nRow > 0)
			if (m_nGrid[nCol][nRow - 1] == nBlock)
				removeBlock(nCol, nRow - 1);
		// Check right
		if (nCol < (clpsNUMCOLS - 1))
			if (m_nGrid[nCol + 1][nRow] == nBlock)
				removeBlock(nCol + 1, nRow);
		// Check left
		if (nCol > 0)
			if (m_nGrid[nCol - 1][nRow] == nBlock)
				removeBlock(nCol - 1, nRow);
	}

// *** PLAY RELATED ***

	// verifies chain, then (if exists) removes.
	int removeChain(int nCol, int nRow) {
		byte nBlock;
		int nIsChain = 0;
		// First, safty check to prevent infinite recursion
		if (m_nGrid[nCol][nRow] == 0)
			return 0;

		// Check neghbors to verify at least one is linked
		nBlock = m_nGrid[nCol][nRow];
		// Check above
		if (nRow < (clpsNUMROWS - 1))
			if (m_nGrid[nCol][nRow + 1] == nBlock)
				nIsChain = 1;
		// Check below
		if (nRow > 0)
			if (m_nGrid[nCol][nRow - 1] == nBlock)
				nIsChain = 1;
		// Check right
		if (nCol < (clpsNUMCOLS - 1))
			if (m_nGrid[nCol + 1][nRow] == nBlock)
				nIsChain = 1;
		// Check left
		if (nCol > 0)
			if (m_nGrid[nCol - 1][nRow] == nBlock)
				nIsChain = 1;

		if (nIsChain > 0)
			removeBlock(nCol, nRow);

		return nIsChain;
	}

	// Drops dangling blocks down ONE row
	// returns lowest dangling block (before drop)
	int collapseColumn(int nCol) {
		boolean nInGap = false;
		int cntrR, nDangle = 0;

		// See if column requires collapsing
		for (cntrR = 0; cntrR < clpsNUMROWS; ++cntrR) {
			if (m_nGrid[nCol][cntrR] == 0)
				nInGap = true;
			else
				if (nInGap) {
					nDangle = cntrR;
					break;
				}
		}

		// Collapse column if necessary
		if (nDangle > 0) {
			for (cntrR = nDangle; cntrR < clpsNUMROWS; ++cntrR)
				m_nGrid[nCol][cntrR - 1] = m_nGrid[nCol][cntrR];
			m_nGrid[nCol][clpsNUMROWS - 1] = 0;
		}
		return nDangle;
	}

	// Returns the row of the topmost block
	int getTopBlock(int nCol) {
		int cntrR, nTop = 0;

		for (cntrR = 0; cntrR < clpsNUMROWS; ++cntrR)
			if (m_nGrid[nCol][cntrR] > 0)
				nTop = cntrR;
		return nTop;
	}

	// Returns the number of blocks that still need to be removed.
	int countBlocks() {
		int cntrC, cntrR, nCount = 0;

		for (cntrC = 0; cntrC < clpsNUMCOLS; ++cntrC)
			for (cntrR = 0; cntrR < clpsNUMROWS; ++cntrR)
				if (m_nGrid[cntrC][cntrR] > 0)
					++nCount;
		return nCount;
	}

// *** VIEW RELATED ***
	// Returns the indicated block
	int getBlock(int nCol, int nRow) {
		return m_nGrid[nCol][nRow];
	}


// *** FILE OPERATIONS ***
	void readFile(DataInputStream iFile) {
		int cntrC, cntrR;

		try {
			for (cntrC = 0; cntrC < clpsNUMCOLS; ++cntrC)
				for (cntrR = 0; cntrR < clpsNUMROWS; ++cntrR)
					m_nGrid[cntrC][cntrR] = iFile.readByte();
		} catch (IOException e) {
			System.out.println("IO Error");
		}
		storeGrid();
	}

}
