/* Dozen Days of Words episode 11: Circle Word (HTML 5)
 * Developed by Billy D. Spelchan
 * Copyright (c) 2010 Blazing Games Inc.
 * based on code from
 * Ultimate Retro Project Episode 46: Modern Circle Word
 * Copyright (C) 2006 Blazing Games Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// *********************
// ***** CONSTANTS *****
// *********************

var FLAG_HORIZONTAL = 1;
var FLAG_DIAG_DOWN = 2;
var FLAG_VERTICAL = 4;
var FLAG_DIAG_UP = 8;

var DIR_HORIZONTAL = 0;
var DIR_DIAG_DOWN = 1;
var DIR_VERTICAL = 2;
var DIR_REVDIAG_DOWN = 3;
var DIR_BACKWARDS = 4;
var DIR_REVDIAG_UP = 5;
var DIR_UPSIDEDOWN = 6;
var DIR_DIAG_UP = 7;

var CANVAS_WIDTH = 640;
var CANVAS_HEIGHT = 480;
var BOARD_BACKGROUND_COLOR = 'rgb(0,0,64)';
var PLAYFIELD_BACKGROUND_COLOR = 'rgb(128,128,255)';
var PLAYFIELD_TEXT_COLOR = 'rgb(64,0,92)';
var CIRCLE_COLOR = 'rgb(92,0,128)';
var PLAYER_CIRCLE_COLOR = 'rgb(192,0,0)';
var WORDBAR_BACKGROUND_COLOR = 'rgb(128,255,128)';
var WORDBAR_TEXT_COLOR = 'rgb(0,64,0)';

// *************************************
// ***** General purpose functions *****
// *************************************

/* create an array containing a randomized order the length of the list */
function randomOrder(n)
{
	var result = new Array(n);
	var swap, temp, cntr;
	
	for (cntr = 0; cntr < n; ++cntr)
		result[cntr] = cntr;
	
	for (cntr = 0; cntr < n; ++cntr)
	{
		swap = Math.floor(Math.random() * n);
		temp = result[swap];
		result[swap] = result[cntr];
		result[cntr] = temp;
	}
	
	return result;
}

// ***********************************
// ***** Circle Word Point Class *****
// ***********************************

function Point(x, y)
{
	this.x = x;
	this.y = y;
}


// **********************************
// ***** Circle Word Rect Class *****
// **********************************

function Rect(x,y,w,h)
{
	this.x = x;
	this.y = y;
	this.width = w;
	this.height = h;
	this.getX2 = function() { return this.x + this.width; }
	this.getY2 = function() { return this.y + this.height; }
	
}

// **************************************
// ***** Circle Word WordInfo Class *****
// **************************************

function WordInfo(s)
{
	this.word = s;
	this.found = false;
	this.start = null;
	this.end = null
}

// **************************************
// ***** Circle Word ColorSet class *****
// **************************************

function ColorSet(board, backg,txt, normCirc, playerCirc, wbar, wbarText)
{
	this.border = board;
	this.background = backg;
	this.text = txt;
	this.circle = normCirc;
	this.playerCircle = playerCirc;
	this.wordbar = wbar;
	this.wordbarText = wbarText;
}

var DEFAULT_COLORSET = new ColorSet(BOARD_BACKGROUND_COLOR, PLAYFIELD_BACKGROUND_COLOR, PLAYFIELD_TEXT_COLOR, CIRCLE_COLOR, PLAYER_CIRCLE_COLOR, WORDBAR_BACKGROUND_COLOR, WORDBAR_TEXT_COLOR);

// ***************************************
// ***** Circle Word Generator Class *****
// ***************************************

function CircleWordGenerator_getRow(row)
{
	var cntr;
	var sb = "";
	
	for (cntr = 0; cntr < this._width; ++cntr)
	{
		sb += this._grid[cntr][row];
	}

	return sb;
}

function CircleWordGenerator_checkPlacement(word, x, y, dir)
{
	var cntr, temp;
	var len = word.length;
	var score = 0;
	
	switch (dir)
	{
		case DIR_HORIZONTAL :
			if ( (x+len) <= this._width)
			{
				for (cntr = 0; cntr < len; ++cntr)
				{
					temp = this._grid[cntr+x][y];
					if (temp != " ")
					{
						if (word.charAt(cntr) == temp)
							++score;
						else
							return -1;
					}
					if ((this._directionFlags[cntr+x][y] & FLAG_HORIZONTAL)==FLAG_HORIZONTAL)
						return -1;
				}
			}
			else
				return -1;
			break;
			
		case DIR_DIAG_DOWN :
			if ( ((x+len) <= this._width) && ((y+len) <= this._height) )
			{
				for (cntr = 0; cntr < len; ++cntr)
				{
					temp = this._grid[cntr+x][y+cntr];
					if (temp != " ")
					{
						if (word.charAt(cntr) == temp)
							++score;
						else
							return -1;
					}
					if ((this._directionFlags[cntr+x][y+cntr] & FLAG_DIAG_DOWN)==FLAG_DIAG_DOWN)
						return -1;
				}
			}
			else
				return -1;
			break;

		case DIR_VERTICAL :
			if ( (y+len) <= this._height) 
			{
				for (cntr = 0; cntr < len; ++cntr)
				{
					temp = this._grid[x][y+cntr];
					if (temp != " ")
					{
						if (word.charAt(cntr) == temp)
							++score;
						else
							return -1;
					}
					if ((this._directionFlags[x][y+cntr] & FLAG_VERTICAL)==FLAG_VERTICAL)
						return -1;

				}
			}
			else
				return -1;
			break;

		case DIR_REVDIAG_DOWN :
			if ( ((x-len) >= 0) && ((y+len) <= this._height) )
			{
				for (cntr = 0; cntr < len; ++cntr)
				{
					temp = this._grid[x-cntr][y+cntr];
					if (temp != " ")
					{
						if (word.charAt(cntr) == temp)
							++score;
						else
							return -1;
					}
					if ((this._directionFlags[x-cntr][y+cntr] & FLAG_DIAG_UP)==FLAG_DIAG_UP)
						return -1;
				}
			}
			else
				return -1;
			break;

		case DIR_BACKWARDS :
			if ( (x-len) >= 0)
			{
				for (cntr = 0; cntr < len; ++cntr)
				{
					temp = this._grid[x-cntr][y];
					if (temp != " ")
					{
						if (word.charAt(cntr) == temp)
							++score;
						else
							return -1;
					}
					if ((this._directionFlags[x-cntr][y] & FLAG_HORIZONTAL)==FLAG_HORIZONTAL)
						return -1;
				}
			}
			else
				return -1;
			break;

		case DIR_REVDIAG_UP :
			if ( ((x-len) >= 0) && ((y-len) >= 0) )
			{
				for (cntr = 0; cntr < len; ++cntr)
				{
					temp = this._grid[x-cntr][y-cntr];
					if (temp != " ")
					{
						if (word.charAt(cntr) == temp)
							++score;
						else
							return -1;
					}
					if ((this._directionFlags[x-cntr][y-cntr] & FLAG_DIAG_DOWN)==FLAG_DIAG_DOWN)
						return -1;
				}
			}
			else
				return -1;
			break;

		case DIR_UPSIDEDOWN :
			if ( (y-len) >= 0)
			{
				for (cntr = 0; cntr < len; ++cntr)
				{
					temp = this._grid[x][y-cntr];
					if (temp != " ")
					{
						if (word.charAt(cntr) == temp)
							++score;
						else
							return -1;
					}
					if ((this._directionFlags[x][y-cntr] & FLAG_VERTICAL)==FLAG_VERTICAL)
						return -1;
				}
			}
			else
				return -1;
			break;

		case DIR_DIAG_UP :
			if ( ((x+len) <= this._width) && ((y-len) >= 0) )
			{
				for (cntr = 0; cntr < len; ++cntr)
				{
					temp = this._grid[cntr+x][y-cntr];
					if (temp != " ")
					{
						if (word.charAt(cntr) == temp)
							++score;
						else
							return -1;
					}
					if ((this._directionFlags[x+cntr][y-cntr] & FLAG_DIAG_UP)==FLAG_DIAG_UP)
						return -1;
				}
			}
			else
				return -1;
			break;
	}
	
	return score;
}

function CircleWordGenerator_layoutWord(word, x, y, dir)
{
	var cntr;
	var len = word.length;
	
	switch (dir)
	{
		case DIR_HORIZONTAL :
			for (cntr = 0; cntr < len; ++cntr)
			{
				this._grid[cntr+x][y] = word.charAt(cntr);
				this._directionFlags[cntr+x][y] |= FLAG_HORIZONTAL;
			}
			break;
			
		case DIR_DIAG_DOWN :
			for (cntr = 0; cntr < len; ++cntr)
			{
				this._grid[cntr+x][cntr+y] = word.charAt(cntr);
				this._directionFlags[cntr+x][cntr+y] |= FLAG_DIAG_DOWN;
			}
			break;

		case DIR_VERTICAL :
			for (cntr = 0; cntr < len; ++cntr)
			{
				this._grid[x][cntr+y] = word.charAt(cntr);
				this._directionFlags[x][cntr+y] |= FLAG_VERTICAL;
			}
			break;

		case DIR_REVDIAG_DOWN :
			for (cntr = 0; cntr < len; ++cntr)
			{
				this._grid[x-cntr][cntr+y] = word.charAt(cntr);
				this._directionFlags[x-cntr][cntr+y] |= FLAG_DIAG_UP;
			}
			break;

		case DIR_BACKWARDS :
			for (cntr = 0; cntr < len; ++cntr)
			{
				this._grid[x-cntr][y] = word.charAt(cntr);
				this._directionFlags[x-cntr][y] |= FLAG_HORIZONTAL;
			}
			break;

		case DIR_REVDIAG_UP :
			for (cntr = 0; cntr < len; ++cntr)
			{
				this._grid[x-cntr][y-cntr] = word.charAt(cntr);
				this._directionFlags[x-cntr][y-cntr] |= FLAG_DIAG_DOWN;
			}
			break;

		case DIR_UPSIDEDOWN :
			for (cntr = 0; cntr < len; ++cntr)
			{
				this._grid[x][y-cntr] = word.charAt(cntr);
				this._directionFlags[x][y-cntr] |= FLAG_VERTICAL;
			}
			break;

		case DIR_DIAG_UP :
			for (cntr = 0; cntr < len; ++cntr)
			{
				this._grid[x+cntr][y-cntr] = word.charAt(cntr);
				this._directionFlags[x+cntr][y-cntr] |= FLAG_DIAG_UP;
			}
			break;
	}
}


function CircleWordGenerator_buildGrid(width, height, num)
{
	this._width = width;
	this._height = height;
	var cntr, cntrR, cntrC, cntrD, temp, count, bestX, bestY, bestD, bestValue;
	var word;
	var usedWords = new Array();
	this._grid = new Array(width);
	this._directionFlags = new Array(width)
	for (cntr = 0; cntr < width; ++cntr) {
		this._grid[cntr] = new Array(height);
		this._directionFlags[cntr] = new Array(height);
	}
	
	bestX = bestY = bestD = -1;
	// first, fill will blanks
	for (cntrR = 0; cntrR < height; ++cntrR)
	{
		for (cntrC = 0; cntrC < width; ++cntrC)
		{
			this._grid[cntrC][cntrR] = " ";
			this._directionFlags[cntrC][cntrR] = 0;
		}
	}

	// To keep things from being lumped, we will validate in random order ...
	// ... so we need to build order lists
	var xOrder = randomOrder(width);
	var yOrder = randomOrder(height);
	var dOrder;
	var wordOrder = randomOrder(this._list.length);

	// now, go through words and add them to grid
	cntr = 0;
	count = 0;
	do
	{
		dOrder = randomOrder(8);
		bestValue = -1;
		word = this._list[wordOrder[cntr]];

		for (cntrR = 0; cntrR < this._height; ++cntrR)
		{
			for (cntrC = 0; cntrC < this._width; ++cntrC)
			{
				for (cntrD = 0; cntrD < 8; ++cntrD)
				{
					temp = this.checkPlacement(word, xOrder[cntrC], yOrder[cntrR], dOrder[cntrD]);
					if (temp > bestValue)
					{
						bestX = xOrder[cntrC];
						bestY = yOrder[cntrR];
						bestD = dOrder[cntrD];
						bestValue = temp;
					}
				}
			}
		}
		if (bestValue >= 0)
		{
			this.layoutWord(word, bestX, bestY, bestD);
			usedWords.push(word);
		}
		else
		{
		}
		++cntr;
	}
	while ( (count < num) && (cntr < this._list.length) );
	
	// next, fill in the blanks
	for (cntrR = 0; cntrR < this._height; ++cntrR)
	{
		for (cntrC = 0; cntrC < this._width; ++cntrC)
		{
			if (this._grid[cntrC][cntrR] == " ")
			{
				this._grid[cntrC][cntrR] = String.fromCharCode( Math.floor(65 + ( Math.random() * 26)));
			}
		}
	}
	
	// finally, create the official word list
	this._words = new Array(usedWords.length);
	for (cntr = 0; cntr < usedWords.length; ++cntr)
		this._words[cntr] = usedWords[cntr];
}

/* ***** CONSTRUCTORs ***** */

function CircleWordGenerator()
{
	// ***** VARIABLES *****
	this._grid = null;
	this._list = new Array();
	this._words = null;
	this._width = 1;
	this._height = 1;
	this._numWords = 1;
	
	this.setWordList = function(list) { this._list = list; }
	this.getLetter = function(x, y) { return this._grid[x][y]; }
	this.getRow = CircleWordGenerator_getRow;
	this.buildGrid = CircleWordGenerator_buildGrid;
	this.checkPlacement = CircleWordGenerator_checkPlacement;
	this.layoutWord = CircleWordGenerator_layoutWord;
	this.getWordsUsed = function() { return this._words; }
}


// **********************************
// ***** Circle Word Game Class *****
// **********************************

// ***** METHODS - Rendering aids *****
function CircleWordGame_circleWord(ctx, start, end)
{
	var pstart = new Point(start.x, start.y);
	var pend = new Point(end.x, end.y);
	
	if (start.x > end.x) { // sort on x position
		var temp = pstart;
		pstart = pend;
		pend = temp;
	}
	var xdist = Math.abs(start.x - end.x);
	var ydist = Math.abs(start.y - end.y);
	var tsx = pstart.x * this._boxWidth + this._playfield.x;
	var tsy = pstart.y * this._boxHeight + this._playfield.y;
	var tex = pend.x * this._boxWidth + this._playfield.x;
	var tey = pend.y * this._boxHeight + this._playfield.y;
	var halfWidth = this._boxWidth / 2;
	var halfHeight = this._boxWidth / 2;
	
	if (xdist == 0) {	// vertical
		if (pstart.y < pend.y) { // two possible directions
			ctx.beginPath();
			ctx.arc(tsx+halfWidth, tsy+halfHeight, halfWidth, 0, Math.PI, true);
			ctx.moveTo(tsx, tsy+halfHeight);
			ctx.lineTo(tsx, tsy+halfHeight+this._boxHeight * ydist);
			ctx.arc(tex+halfWidth, tey+halfHeight, halfWidth, Math.PI, 2*Math.PI, true);
			ctx.moveTo(tsx + this._boxWidth, tsy+halfHeight);
			ctx.lineTo(tsx + this._boxWidth, tsy+halfHeight+this._boxHeight * ydist);
			ctx.stroke();
		} else {
			ctx.beginPath();
			ctx.arc(tex+halfWidth, tey+halfHeight, halfWidth, 0, Math.PI, true);
			ctx.moveTo(tex, tey+halfHeight);
			ctx.lineTo(tex, tey+halfHeight+this._boxHeight * ydist);
			ctx.arc(tsx+halfWidth, tsy+halfHeight, halfWidth, Math.PI, 2*Math.PI, true);
			ctx.moveTo(tex + this._boxWidth, tey+halfHeight);
			ctx.lineTo(tex + this._boxWidth, tey+halfHeight+this._boxHeight * ydist);
			ctx.stroke();
		}
	} else if (ydist == 0) {	// horizontal
		ctx.beginPath();
		ctx.arc(tsx+halfWidth, tsy+halfHeight, halfWidth, 1.5*Math.PI, .5*Math.PI, true);
		ctx.moveTo(tsx+halfWidth, tsy);
		ctx.lineTo(tsx+halfWidth + this._boxWidth * xdist, tsy);
		ctx.arc(tex+halfWidth, tey+halfHeight, halfWidth, 1.5*Math.PI, .5*Math.PI, false);
		ctx.moveTo(tsx + halfWidth + this._boxWidth * xdist, tsy+this._boxHeight);
		ctx.lineTo(tsx + halfWidth, tsy+this._boxHeight);
		ctx.stroke();

	} else if (xdist == ydist) { // diagonal

		var pct15Width = this._boxWidth * 15 / 100;
		var pct15Height = this._boxHeight * 15 / 100;
		var pct85Width = this._boxWidth - pct15Width;
		var pct85Height = this._boxHeight - pct15Height;
		if (pstart.y < pend.y) { // two possible directions - downward
			ctx.beginPath();
			ctx.arc(tsx+halfWidth, tsy+halfHeight, halfWidth, 1.75*Math.PI, .75*Math.PI, true);
			ctx.moveTo(tsx + pct15Width, tsy +pct85Height);
			ctx.lineTo(tex + pct15Width, tey + pct85Height);
			ctx.arc(tex + halfWidth, tey + halfHeight, halfWidth, .75*Math.PI, 1.75*Math.PI, true);
			ctx.moveTo(tex + pct85Width, tey + pct15Height);
			ctx.lineTo(tsx + pct85Width, tsy + pct15Height);
			ctx.stroke();
		} else { // upward
			ctx.beginPath();
			ctx.arc(tsx + halfWidth, tsy + halfHeight, halfWidth, 1.25 * Math.PI, .25 * Math.PI, true);
			ctx.moveTo(tsx + pct85Width, tsy + pct85Height);
			ctx.lineTo(tex + pct85Width, tey + pct85Height);
			ctx.arc(tex + halfWidth, tey + halfHeight, halfWidth, .25 * Math.PI, 1.25 * Math.PI, true);
			ctx.moveTo(tex + pct15Width, tey + pct15Height);
			ctx.lineTo(tsx + pct15Width, tsy + pct15Height);
			ctx.stroke();
		}
	}
	
}

function CircleWordGame_render(ctx)
{
	var cntrRow, cntrCol, offX, offY;
	var temp;

	ctx.fillStyle = BOARD_BACKGROUND_COLOR;
	ctx.fillRect(0,0, CANVAS_WIDTH, CANVAS_HEIGHT);

	var w = this._generator._width;
	var h = this._generator._height;

	ctx.fillStyle = PLAYFIELD_BACKGROUND_COLOR;
	ctx.fillRect(this._playfield.x, this._playfield.y, this._playfield.width, this._playfield.height);
	
	ctx.fillStyle = PLAYFIELD_TEXT_COLOR;
	ctx.font = (this._boxWidth - 4) + "px MonoSpace";
	ctx.textBaseline = "top";

	offY = 2;
	offX = 2;
	for (cntrRow = 0; cntrRow < h; ++cntrRow)
	{            
		for (cntrCol = 0; cntrCol < w; ++cntrCol)
		{
			temp = this._generator.getLetter(cntrCol, cntrRow);
			ctx.fillText(temp, this._playfield.x + offX + cntrCol * this._boxWidth,	this._playfield.y + offY + cntrRow * this._boxHeight);
		}
	}
	
	// draw the word bar
	ctx.fillStyle = WORDBAR_BACKGROUND_COLOR;
	ctx.fillRect(this._wordbar.x, this._wordbar.y, this._wordbar.width, this._wordbar.height);
	listHeight = this._wordbar.height / this._wordsInPuzzle.length;
	ctx.font = (listHeight-2)+"px MonoSpace";
	ctx.fillStyle = WORDBAR_TEXT_COLOR;
	ctx.strokeStyle = CIRCLE_COLOR;
	var wip;
	for (cntrRow = 0; cntrRow < this._wordsInPuzzle.length; ++cntrRow) {
		wip = this._wordsInPuzzle[cntrRow];
		ctx.fillText(wip.word, this._wordbar.x + 5, this._wordbar.y + cntrRow * listHeight);
		if (wip.found == true) {
			var lineY = this._wordbar.y + cntrRow * listHeight + listHeight / 2;
			ctx.beginPath();
			ctx.moveTo(this._wordbar.x, lineY);
			ctx.lineTo(this._wordbar.x + this._wordbar.width, lineY);
			ctx.stroke();
			this.circleWord(ctx, wip.start, wip.end);
		}
	}
		
	// Draw the current user selection
	ctx.strokeStyle = PLAYER_CIRCLE_COLOR;
	if (this._selected.x >= 0)
	{
		this.circleWord(ctx, this._currentTile, this._selected);
	}
	
}

function CircleWordGame_getPuzzlePoint(x, y)
{
	var tx = Math.floor((x - this._playfield.x) / this._boxWidth);
	var ty = Math.floor((y - this._playfield.y) / this._boxHeight);
	if ((tx < 0) || (ty < 0) || (tx >= this._generator._width) || (ty >= this._generator._height)) {
		tx = -1;
		ty = -1;
	}
	return new Point(tx,ty);
}

function CircleWordGame_mouseMove(x, y)
{
	var puzPoint = this.getPuzzlePoint(x, y);
	if ( (this._currentTile.x != puzPoint.x) || (this._currentTile.y != puzPoint.y) )
	{
		this._currentTile.x = puzPoint.x;
		this._currentTile.y = puzPoint.y;
		return true;
	} else 
		return false;
}

function CircleWordGame_mouseDown(x, y)
{
	this._selected = this.getPuzzlePoint(x, y);
}

function CircleWordGame_mouseUp(x, y)
{
	if (this._selected.x < 0) return;
	var puzPoint = this.getPuzzlePoint(x, y);
	var dx = puzPoint.x - this._selected.x;
	var dy = puzPoint.y - this._selected.y;
	if ((dx == 0) || (dy == 0) || (Math.abs(dx) == Math.abs(dy))) {
		var sb = "";
		var dirX, dirY;
		var curX = this._selected.x;
		var curY = this._selected.y;
		if (dx < 0)
			dirX = -1;
		else if (dx == 0)
			dirX = 0;
		else
			dirX = 1;
		if (dy < 0)
			dirY = -1;
		else if (dy == 0)
			dirY = 0;
		else
			dirY = 1;
		var targetLen = Math.max(Math.abs(dx), Math.abs(dy));
		var curLen = 0;
		while (curLen <= targetLen) {
			sb += (this._generator.getLetter(curX, curY));
			curX += dirX;
			curY += dirY;
			++curLen;
		};
		var word = sb;
		var revword = "";
		var cntr;
		for (cntr = 0; cntr < sb.length; ++cntr)
			revword += sb.charAt(sb.length - 1 - cntr);

		var wip;
		for (cntr = 0; cntr < this._wordsInPuzzle.length; ++cntr) {
			wip = this._wordsInPuzzle[cntr];
			if ((wip.found == false) &&
					( (wip.word == word) || 
					(wip.word == revword) ) ) {
				wip.found = true;
				wip.start = new Point(this._selected.x, this._selected.y);
				wip.end = puzPoint;
				break;
			}
		}
	}
	
	this._selected.x = -1;
	this._selected.y = -1;
}

function CircleWordGame_startNewGame(w,h,n,wl,cs)
{
	this._generator.setWordList(wl);
	this._generator.buildGrid(w,h,n);
	BOARD_BACKGROUND_COLOR = cs.border;
	PLAYFIELD_BACKGROUND_COLOR = cs.background;
	PLAYFIELD_TEXT_COLOR = cs.text;
	CIRCLE_COLOR = cs.circle;
	PLAYER_CIRCLE_COLOR = cs.playerCircle;
	WORDBAR_BACKGROUND_COLOR = cs.wordbar;
	WORDBAR_TEXT_COLOR = cs.wordbarText;

	var temp = this._generator.getWordsUsed();
	if (temp == null) return;
	this._wordsInPuzzle = new Array();
	for (var cntr = 0; cntr < temp.length; ++cntr)
		this._wordsInPuzzle.push(new WordInfo(temp[cntr]));
	this._currentTile = new Point(-1,-1);
	this._selected = new Point(-1,-1);
	this._boxWidth = Math.floor(this._playfield.width / cwg._width);
	this._boxHeight = Math.floor(this._playfield.height / cwg._height);
}

// ***** Constructor *****
function CircleWordGame(cwg)
{
	this._generator = cwg;
	this.startNewGame = CircleWordGame_startNewGame;
	this._playfield = new Rect(15,15,450,450);
	this._wordbar = new Rect(480,0,160,480);
	this.circleWord = CircleWordGame_circleWord;
	this.render = CircleWordGame_render;
	
	this.getPuzzlePoint = CircleWordGame_getPuzzlePoint;
	this.mouseMove = CircleWordGame_mouseMove;
	this.mouseDown = CircleWordGame_mouseDown;
	this.mouseUp = CircleWordGame_mouseUp;
}

