/* Dozen Days of Words episode 12: Morse Code (HTML 5)
 * Developed by Billy D. Spelchan
 * Copyright (c) 2010 Blazing Games Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// *********************
// ***** CONSTANTS *****
// *********************

var CANVAS_WIDTH = 640;
var CANVAS_HEIGHT = 480;
var MORSECODE_TITLE = 0;
var MORSECODE_ENCODE = 1;
var MORSECODE_DECODE = 2;

/*
var BOARD_BACKGROUND_COLOR = 'rgb(0,0,64)';
var PLAYFIELD_BACKGROUND_COLOR = 'rgb(128,128,255)';
var PLAYFIELD_TEXT_COLOR = 'rgb(64,0,92)';
var CIRCLE_COLOR = 'rgb(92,0,128)';
var PLAYER_CIRCLE_COLOR = 'rgb(192,0,0)';
var WORDBAR_BACKGROUND_COLOR = 'rgb(128,255,128)';
var WORDBAR_TEXT_COLOR = 'rgb(0,64,0)';
*/

var DOT = 0;
var DASH = 1;
var CODE_ORDER = "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890";
var CODES = new Array(
			new Array(DOT, DASH), // A
			new Array(DASH, DOT, DOT, DOT), // B
			new Array(DASH, DOT, DASH, DOT), // C
			new Array(DASH, DOT, DOT), // D
			new Array(), // E -> single length so have to manually create later
			new Array(DOT, DOT, DASH, DOT), // F
			new Array(DASH, DASH, DOT), // G
			new Array(DOT, DOT, DOT, DOT), // H
			new Array(DOT, DOT), // I
			new Array(DOT, DASH, DASH, DASH), // J
			new Array(DASH, DOT, DASH), // K
			new Array(DOT, DASH, DOT, DOT), // L
			new Array(DASH, DASH), // M
			new Array(DASH, DOT), // N
			new Array(DASH, DASH, DASH), // O
			new Array(DOT, DASH, DASH, DOT), // P
			new Array(DASH, DASH, DOT, DASH), // Q
			new Array(DOT, DASH, DOT), // R
			new Array(DOT, DOT, DOT), // S
			new Array(), // T -> single length so have to manually create later
			new Array(DOT, DOT, DASH), // U
			new Array(DOT, DOT, DOT, DASH), // V
			new Array(DOT, DASH, DASH), // W
			new Array(DASH, DOT, DOT, DASH), // X
			new Array(DASH, DOT, DASH, DASH), // Y
			new Array(DASH, DASH, DOT, DOT), // Z
			new Array(DOT, DASH, DASH, DASH, DASH), // 1
			new Array(DOT, DOT, DASH, DASH, DASH), // 2
			new Array(DOT, DOT, DOT, DASH, DASH), // 3
			new Array(DOT, DOT, DOT, DOT, DASH), // 4
			new Array(DOT, DOT, DOT, DOT, DOT), // 5
			new Array(DASH, DOT, DOT, DOT, DOT), // 6
			new Array(DASH, DASH, DOT, DOT, DOT), // 7
			new Array(DASH, DASH, DASH, DOT, DOT), // 8
			new Array(DASH, DASH, DASH, DASH, DOT), // 9
			new Array(DASH, DASH, DASH, DASH, DASH) // 0
		);
CODES[4].push(DOT); // E corrected
CODES[19].push(DASH); // T corrected
// *************************************
// ***** General purpose functions *****
// *************************************

/* create an array containing a randomized order the length of the list */
function randomOrder(n)
{
	var result = new Array(n);
	var swap, temp, cntr;
	
	for (cntr = 0; cntr < n; ++cntr)
		result[cntr] = cntr;
	
	for (cntr = 0; cntr < n; ++cntr)
	{
		swap = Math.floor(Math.random() * n);
		temp = result[swap];
		result[swap] = result[cntr];
		result[cntr] = temp;
	}
	
	return result;
}

function inheritProperties(child, supr)
{
	for (var property in supr) {  
		if (typeof child[property] == "undefined")  
			child[property] = supr[property];  
	}  
}

// *************************
// ***** BGPoint Class *****
// *************************

function BGPoint(x, y)
{
	this.x = x;
	this.y = y;
}

// *****************************
// ***** BGDimension Class *****
// *****************************

function BGDimension(w, h)
{
	this.width = w;
	this.height = h;
}

// ************************
// ***** BGRect Class *****
// ************************

function BGRect(x,y,w,h)
{
	this.x = x;
	this.y = y;
	this.width = w;
	this.height = h;
}

BGRect.prototype.getX2 = function() { return this.x + this.width; }

BGRect.prototype.getY2 = function() { return this.y + this.height; }

BGRect.prototype.contains = function(x,y) {
	var deltaX = x - this.x;
	var deltaY = y - this.y;
	if ((deltaX < 0) || (deltaY < 0) || (deltaX > this.width) || (deltaY > this.height))
		return false;
	else
		return true;
}

// *************************
// ***** BGLayer Class *****
// *************************

function BGLayer(id, rect)
{
	this.id = id;
	this.placement = new BGRect(rect.x, rect.y, rect.width, rect.height);
	this.solid = true;
	this.backgroundColor = 'rgb(0,0,0)';
	this.children = new Array();
	this.visible = true;
}
	
BGLayer.prototype.addChild = function(child) { this.children.push(child); };
	
BGLayer.prototype.removeChild = function(child)
{
	var childIndex = -1;
	var cntr;
	
	for (cntr = 0; cntr < this.children.length; ++cntr)
		if (this.children[cntr] == child)
			childIndex = cntr;
	if (childIndex >= 0)
		this.children.splice(childIndex, 1);
}

BGLayer.prototype.drawSelf = function(ctx)
{
	if (this.solid) {
		ctx.save();
		var oldFill = ctx.fillStyle;
		ctx.fillStyle = this.backgroundColor;
		ctx.fillRect(this.placement.x, this.placement.y, this.placement.width, this.placement.height);
		ctx.fillStyle =oldFill;
		ctx.restore();
	}	
}

BGLayer.prototype.render = function(ctx)
{
	if (!this.visible)
		return;
	this.drawSelf(ctx);
	for (var cntr = 0; cntr < this.children.length; ++cntr)
		this.children[cntr].render(ctx);
}

BGLayer.prototype.mouseMove = function(x, y)
{
	var dirty = false;
	for (var cntr = 0; cntr < this.children.length; ++cntr)
		dirty |= this.children[cntr].mouseMove(x,y);
	return dirty;
}

BGLayer.prototype.mouseDown = function(x, y)
{
	var dirty = false;
	for (var cntr = 0; cntr < this.children.length; ++cntr)
		dirty |= this.children[cntr].mouseDown(x,y);
	return dirty;
}

BGLayer.prototype.mouseUp = function(x, y)
{
	var dirty = false;
	for (var cntr = 0; cntr < this.children.length; ++cntr)
		dirty |= this.children[cntr].mouseUp(x,y);
	return dirty;
}


// *************************
// ***** BGLabel Class *****
// *************************

function BGLabel(id, rect, s)
{
	inheritProperties(this, new BGLayer(id, rect));
	this.font = (rect.height - 4) + "px sans-serif";
	this.text = s;
	this.textColor = "black";
}

BGLabel.prototype.setText = function(s) { this.text = s; };

BGLabel.prototype.setTextColor = function(c) { this.textColor = c; };

BGLabel.prototype.drawSelf = function(ctx)
{
	ctx.save();
	var oldFill = ctx.fillStyle;
	ctx.fillStyle = this.backgroundColor;
	ctx.fillRect(this.placement.x, this.placement.y, this.placement.width, this.placement.height);
	
	ctx.fillStyle = this.textColor;
	var oldFont = ctx.font;
	ctx.font = this.font;
	ctx.textBaseline = "top";
	ctx.textAlign = "center";
	ctx.fillText(this.text, this.placement.x + this.placement.width / 2, this.placement.y, this.placement.width);
	ctx.font = oldFont;
	
	ctx.fillStyle =oldFill;
	ctx.restore();
}


// **************************
// ***** BGButton Class *****
// **************************

function BGButton(id, rect, s)
{
	inheritProperties(this, new BGLayer(id, rect, s));
	this.stateUpBackColor = "blue";
	this.stateOverBackColor = "cyan";
	this.stateDownBackColor = "red";
	this.textColor = "white";
	this.label = new BGLabel(id, rect, s);
	this.buttonState = 0; // up
}

BGButton.prototype.drawSelf = function(ctx)
{
	if ( !this.visible )
		return;
	this.label.textColor = this.textColor;
	switch (this.buttonState) {
		case 0: // up
			this.label.backgroundColor = this.stateUpBackColor;
			break;
		case 1: // over
			this.label.backgroundColor = this.stateOverBackColor;
			break;
		case 2: // down
			this.label.backgroundColor = this.stateDownBackColor;
			break;
	}
	
	this.label.drawSelf(ctx);
}

BGButton.prototype.mouseMove = function(x, y)
{
	if ( ! this.visible)
		return false;
	var isOver = this.placement.contains(x,y);
	if (isOver) {
		if (this.buttonState > 0)
			return false;
		else {
			this.buttonState = 1;
			return true;
		}
	} else {
		if (this.buttonState == 0)
			return false;
		else {
			this.buttonState = 0;
			return true;
		}
	}
}

BGButton.prototype.mouseDown = function(x, y)
{
	if ( ! this.visible)
		return false;
	var isOver = this.placement.contains(x,y);
	if (isOver)
		this.buttonState = 2;
	return isOver;
}

BGButton.prototype.mouseUp = function(x, y)
{
	if ( ! this.visible)
		return false;
	var isOver = this.placement.contains(x,y);
	if ((isOver) && (this.buttonState == 2) && (this.onClick != null))
		this.onClick.clicked(this.id);
	if (this.buttonState == 0)
		return false;
	this.buttonState = 0;		
	return true;
}

// *********************************
// ***** MorseTile Layer Class *****
// *********************************

function MorseTile(id, rect, morse)
{
	inheritProperties(this, new BGLayer(id, rect));

	this.color = 'rgb(255,255,0)';
	this.backgroundColor = 'rgb(64,64,64)';
	if (morse == null)
		this.dotCode = new Array();
	else
		this.dotCode = morse;

}

MorseTile.prototype.setColors = function(c, b) { this.color = c; this.backcolor = b; };

MorseTile.prototype.setCode = function(morse) { if (morse == null) this.dotCode = new Array(); else this.dotCode = morse; };

MorseTile.prototype.addToCode = function(dotdash) { this.dotCode.push(dotdash); };

MorseTile.prototype.drawSelf = function(ctx) 
{
	var parts = new Array();
	var cntr;

	for (cntr = 0; cntr < this.dotCode.length; ++cntr) {
		parts.push(1);
		if (this.dotCode[cntr] == 1) {
			parts.push(1);
			parts.push(1);
		}
		parts.push(0)
	}
	parts.pop();
			
	var dotSize = Math.min(this.placement.height, this.placement.width / parts.length);
	var dotX = this.placement.x;
	var dotY = this.placement.y;
	ctx.save();
	var oldStyle = ctx.fillStyle;
		ctx.fillStyle = this.backgroundColor;
		ctx.fillRect(this.placement.x, this.placement.y, this.placement.width, this.placement.height);
	ctx.fillStyle = this.color;
	for (cntr = 0; cntr < parts.length; ++cntr) {
		if (parts[cntr] == 1)
			ctx.fillRect(dotX, dotY, dotSize, dotSize);
		dotX += dotSize;
	}
	ctx.restore();
}

// ***********************************
// ***** MorseEncodeLetter Class *****
// ***********************************

function MorseEncodeLetter(game)
{
	inheritProperties(this, new BGLayer("EncodeLetterStage", new BGRect(0,0, CANVAS_WIDTH, CANVAS_HEIGHT)));
	this.game = game;
	this.backgroundColor = "rgb(25,25,112)";
	this.message = new BGLabel("msg", new BGRect(0,0,640,40), "Messages go here");
	this.message.backgroundColor = this.backgroundColor;
	this.message.textColor = "white";
	this.addChild(this.message);
	this.targetLetter = new BGLabel("letter", new BGRect(256,64,128,128), "?");
	this.targetLetter.backgroundColor = this.backgroundColor;
	this.targetLetter.textColor = "rgb(255,215,0)";
	this.addChild(this.targetLetter);
	this.targetEncode = new MorseTile("letter", new BGRect(192,192,256,32), null);
	this.addChild(this.targetEncode);
	this.correctEncode = new MorseTile("letter", new BGRect(192,234,256,32), null);
	this.addChild(this.correctEncode);

	this.dotButton = new BGButton("dot", new BGRect(10,440,200,40), "Dot");
	this.dotButton.onClick = this;
	this.addChild(this.dotButton);
	this.dashButton = new BGButton("dash", new BGRect(220,440,200,40), "Dash");
	this.dashButton.onClick = this;
	this.addChild(this.dashButton);
	this.doneButton = new BGButton("done", new BGRect(440,440,200,40), "Done");
	this.doneButton.onClick = this;
	this.addChild(this.doneButton);

	this.continueButton = new BGButton("continue", new BGRect(20,300,600,50), "continue");
	this.continueButton.onClick = this;
	this.continueButton.visible = false;
	this.addChild(this.continueButton);

}

MorseEncodeLetter.prototype.clicked = function(id)
{
	if (id == "continue") {
		this.nextRound();
		return;
	}
	if (id == "dot")
		this.targetEncode.addToCode(DOT);
	if (id == "dash")
		this.targetEncode.addToCode(DASH);
	if (id == "done") {
		this.correctEncode.setCode(CODES[this.curLetter]);
		var matchingCode = true;
		if (this.targetEncode.dotCode.length == CODES[this.curLetter].length) {
			for (var cntr = 0; cntr < this.targetEncode.dotCode.length; ++cntr)
				if (this.targetEncode.dotCode[cntr] != CODES[this.curLetter][cntr])
					matchingCode = false;
		} else
			var matchingCode = false;
			
		if (matchingCode) {
			this.message.text = "Correct answer";
			++this.score;
		} else
			this.message.text = "Incorrect answer ";
		if (this.gameRound >= 10)
			this.continueButton.label.text = "Score is " + this.score + " out of " + this.gameRound;
		this.continueButton.visible = true;
		this.dotButton.visible = false;
		this.dashButton.visible = false;
		this.doneButton.visible = false;
	}
}



MorseEncodeLetter.prototype.nextRound = function()
{
	++this.gameRound;
	if (this.gameRound > 10) {
		game.changeMode(MORSECODE_TITLE);
		return;
	}	
	this.message.text = "Round " + this.gameRound + " of 10";			
	this.curLetter = Math.floor(Math.random() * CODE_ORDER.length);
	this.targetLetter.text = CODE_ORDER.substr(this.curLetter,1);
	this.targetEncode.setCode(null);
	this.correctEncode.setCode(null);
	this.continueButton.visible = false;
	this.dotButton.visible = true;
	this.dashButton.visible = true;
	this.doneButton.visible = true;
}

MorseEncodeLetter.prototype.startGame = function()
{
	this.continueButton.label.text = "Continue";
	this.gameRound = 0;
	this.score = 0;
	this.nextRound();
}


// ***********************************
// ***** MorseDecodeLetter Class *****
// ***********************************

function MorseDecodeLetter()
{
	inheritProperties(this, new BGLayer("EncodeLetterStage", new BGRect(0,0, CANVAS_WIDTH, CANVAS_HEIGHT)));
	this.backgroundColor = "rgb(0,0,128)";
	this.message = new BGLabel("msg", new BGRect(0,0,640,40), "Messages go here");
	this.message.backgroundColor = this.backgroundColor;
	this.message.textColor = "White";
	this.addChild(this.message);
	this.targetLetter = new BGLabel("letter", new BGRect(256,64,128,128), "?");
	this.targetLetter.backgroundColor = this.backgroundColor;
	this.targetLetter.textColor = "rgb(255,215,0)";
	this.addChild(this.targetLetter);
	this.targetEncode = new MorseTile("letter", new BGRect(64,192,512,32), null);
	this.addChild(this.targetEncode);

	this.letterButtons = new Array();
	var indx = 0;
	var btnX, btnY;
	var btnLetter;
	for (var cntrRow = 0; cntrRow < 4; ++cntrRow) {
		btnY = 250 + 55 * cntrRow;
		btnX = 32;
		for (var cntrCol = 0; cntrCol < 9; ++cntrCol) {
			btnLetter = CODE_ORDER.substr(indx, 1);
			this.letterButtons[indx] = new BGButton(btnLetter, new BGRect(btnX, btnY, 60,50), btnLetter);
			this.letterButtons[indx].onClick = this;
			this.addChild(this.letterButtons[indx]);
			++indx;
			btnX += 64;
		}
	}
	
	this.continueButton = new BGButton("continue", new BGRect(20,224,600,50), "continue");
	this.continueButton.onClick = this;
	this.continueButton.visible = false;
	this.addChild(this.continueButton);
}

MorseDecodeLetter.prototype.clicked = function(id)
{
	if (id == "continue") {
		this.nextRound();
		return;
	}
	if (id == CODE_ORDER.substr(this.curLetter,1)) {
		++this.score;
		this.message.text = "Correct answer";
	} else {
		this.message.text = "Incorrect answer ";			
	}
	this.targetLetter.text = CODE_ORDER.substr(this.curLetter,1);
	
	for (var cntr = 0; cntr < this.letterButtons.length; ++cntr)
		this.letterButtons[cntr].visible = false;
	if (this.gameRound >= 10)
		this.continueButton.label.text = "Score is " + this.score + " out of " + this.gameRound;
	this.continueButton.visible = true;

}

MorseDecodeLetter.prototype.nextRound = function()
{
	++this.gameRound;
	if (this.gameRound > 10) {
		game.changeMode(MORSECODE_TITLE);
		return;
	}	
	this.message.text = "Round " + this.gameRound + " of 10";			
	this.curLetter = Math.floor(Math.random() * CODE_ORDER.length);
	this.targetLetter.text = "?"//CODE_ORDER.substr(this.curLetter,1);
	this.targetEncode.setCode(CODES[this.curLetter]);
	for (var cntr = 0; cntr < this.letterButtons.length; ++cntr)
		this.letterButtons[cntr].visible = true;
	this.continueButton.visible = false;
}

MorseDecodeLetter.prototype.startGame = function()
{
	this.continueButton.label.text = "Continue";
	this.gameRound = 0;
	this.score = 0;
	this.nextRound();
}


// ****************************
// ***** MorseTitle Class *****
// ****************************

function MorseTitle(game)
{
	inheritProperties(this, new BGLayer("TitleStage", new BGRect(0,0, CANVAS_WIDTH, CANVAS_HEIGHT)));
	this.game = game;
	this.backgroundColor = "DarkViolet";
	
	var cntr, tempChar, tempX, tempY
	var tempLayer;
	var tempWord = "MORSE";
	tempX = 0;
	tempY = 20;
	for (cntr = 0; cntr < tempWord.length; ++cntr) {
		tempLayer = new BGLabel(tempWord+cntr, new BGRect(tempX, tempY, 120,120), tempWord.substr(cntr, 1));
		tempLayer.setTextColor("Silver");
		tempLayer.backgroundColor = this.backgroundColor;
		this.addChild(tempLayer);
		tempLayer = new MorseTile(tempWord+cntr, new BGRect(tempX, tempY+120,120,16), CODES[CODE_ORDER.indexOf(tempWord.substr(cntr, 1))]);
		this.addChild(tempLayer);
		tempX += 128;
	}

	var tempWord = "CODE";
	tempX = 64;
	tempY = 160;
	for (cntr = 0; cntr < tempWord.length; ++cntr) {
		tempLayer = new BGLabel(tempWord+cntr, new BGRect(tempX, tempY, 120,120), tempWord.substr(cntr, 1));
		tempLayer.setTextColor("Silver");
		tempLayer.backgroundColor = this.backgroundColor;
		this.addChild(tempLayer);
		tempLayer = new MorseTile(tempWord+cntr, new BGRect(tempX, tempY+120,120,16), CODES[CODE_ORDER.indexOf(tempWord.substr(cntr, 1))]);
		this.addChild(tempLayer);
		tempX += 128;
	}

	this.encodeButton = new BGButton("encode", new BGRect(20,320,600,50), "Letters to Morse Code");
	this.encodeButton.onClick = this;
	this.addChild(this.encodeButton);

	this.decodeButton = new BGButton("decode", new BGRect(20,380,600,50), "Morse Code to Letters");
	this.decodeButton.onClick = this;
	this.addChild(this.decodeButton);
}

MorseTitle.prototype.clicked = function(id)
{
	if (id == "encode")
		this.game.changeMode(MORSECODE_ENCODE);
	if (id == "decode")
		this.game.changeMode(MORSECODE_DECODE);	
}

MorseTitle.prototype.startGame = function()
{
}

// *********************************
// ***** Morse Code Game Class *****
// *********************************

// ***** Constructor *****
function MorseCodeGame()
{
	this.titleStage = new MorseTitle(this);
	this.encodeStage = new MorseEncodeLetter(this);
	this.decodeStage = new MorseDecodeLetter(this);
	this.curMode = MORSECODE_TITLE;
	this.stage = this.titleStage;
	this.stage.startGame();
	
}

MorseCodeGame.prototype.changeMode = function(n)
{
	switch (n) {
		case MORSECODE_TITLE:
			this.stage = this.titleStage;
			this.stage.startGame();
			break;
		
		case MORSECODE_ENCODE:
			this.stage = this.encodeStage;
			this.stage.startGame();
			break;

		case MORSECODE_DECODE:
			this.stage = this.decodeStage;
			this.stage.startGame();
			break;
	}
}

MorseCodeGame.prototype.mouseMove = function(x, y)
{
	return this.stage.mouseMove(x,y);
}

MorseCodeGame.prototype.mouseDown = function(x, y)
{
	return this.stage.mouseDown(x,y);
}

MorseCodeGame.prototype.mouseUp = function(x, y)
{
	return this.stage.mouseUp(x,y)
}


MorseCodeGame.prototype.render = function(ctx, full)
{
	this.stage.render(ctx);
}
